/**
 * Utility functions for Framer Motion animations
 * Provides mobile detection and animation control
 */

/**
 * Checks if the device is a mobile phone
 * @returns {boolean} True if mobile phone, false otherwise
 */
export const isMobilePhone = () => {
  if (typeof window === "undefined") return false;

  // Check screen width (typical mobile phone width)
  const isMobileWidth = window.innerWidth <= 768;

  // Check user agent for mobile devices
  const userAgent = navigator.userAgent || navigator.vendor || window.opera;
  const isMobileDevice =
    /android|webos|iphone|ipod|blackberry|iemobile|opera mini/i.test(
      userAgent.toLowerCase()
    );

  // Return true only if both conditions are met (to exclude tablets in landscape)
  return isMobileWidth && isMobileDevice;
};

/**
 * Checks if the device is a tablet
 * @returns {boolean} True if tablet, false otherwise
 */
export const isTablet = () => {
  if (typeof window === "undefined") return false;

  const userAgent = navigator.userAgent || navigator.vendor || window.opera;
  const isTabletDevice = /ipad|android(?!.*mobile)|tablet/i.test(
    userAgent.toLowerCase()
  );

  const isTabletWidth = window.innerWidth > 768 && window.innerWidth <= 1024;

  return isTabletDevice || isTabletWidth;
};

/**
 * Returns animation variants based on whether animations should be disabled
 * @param {Object} animatedVariants - The variants to use when animations are enabled
 * @param {Object} staticVariants - The variants to use when animations are disabled
 * @param {boolean} disableOnMobile - Whether to disable animations on mobile phones
 * @returns {Object} The appropriate variants object
 */
export const getAnimationVariants = (
  animatedVariants,
  staticVariants,
  disableOnMobile = true
) => {
  if (typeof window === "undefined") return animatedVariants;

  if (disableOnMobile && isMobilePhone()) {
    return staticVariants;
  }

  return animatedVariants;
};

/**
 * Returns animation props that can be spread onto motion components
 * When animations are disabled, returns props that skip animation
 * @param {Object} animationProps - The animation props (initial, animate, transition, etc.)
 * @param {boolean} disableOnMobile - Whether to disable animations on mobile phones
 * @returns {Object} The appropriate props object
 */
export const getMotionProps = (animationProps, disableOnMobile = true) => {
  if (typeof window === "undefined") return animationProps;

  if (disableOnMobile && isMobilePhone()) {
    // Return props that skip animation by making initial same as animate
    const staticProps = {};

    // If there's an animate prop, use it as both initial and animate
    if (animationProps.animate) {
      staticProps.initial = animationProps.animate;
      staticProps.animate = animationProps.animate;
    }

    // If there's a whileInView prop, use it as both initial and animate
    if (animationProps.whileInView) {
      staticProps.initial = animationProps.whileInView;
      staticProps.animate = animationProps.whileInView;
    }

    // Set transition to instant
    staticProps.transition = { duration: 0 };

    return staticProps;
  }

  return animationProps;
};

/**
 * Hook to check if animations should be enabled
 * @param {boolean} disableOnMobile - Whether to disable animations on mobile phones
 * @returns {boolean} True if animations should be enabled
 */
export const shouldAnimate = (disableOnMobile = true) => {
  if (typeof window === "undefined") return true;

  if (disableOnMobile && isMobilePhone()) {
    return false;
  }

  return true;
};

/**
 * Get reduced motion preference from system settings
 * @returns {boolean} True if user prefers reduced motion
 */
export const prefersReducedMotion = () => {
  if (typeof window === "undefined") return false;

  return window.matchMedia("(prefers-reduced-motion: reduce)").matches;
};
