"use client";

import { useEffect, useRef, useState } from "react";
import Link from "next/link";
import Image from "next/image";
import { useRouter } from "next/navigation";
import {
  X,
  Search as SearchIcon,
  FileText,
  Box,
  Tag,
  Loader2,
} from "lucide-react";

export default function SearchPreview({
  query,
  onClose,
  isOpen,
  onSearchSubmit,
}) {
  const [results, setResults] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const searchRef = useRef(null);
  const router = useRouter();

  // Handle keyboard events
  useEffect(() => {
    const handleKeyDown = (e) => {
      if (e.key === "Escape") {
        onClose();
      }
    };

    if (isOpen) {
      document.addEventListener("keydown", handleKeyDown);
      return () => {
        document.removeEventListener("keydown", handleKeyDown);
      };
    }
  }, [isOpen, onClose]);

  // Search functionality with debounce
  useEffect(() => {
    const search = async () => {
      if (!query || query.trim() === "") {
        setResults([]);
        return;
      }

      setIsLoading(true);
      try {
        const { searchAll } = await import("@/queries/searchQueries");
        const searchResults = searchAll(query);
        console.log("Search results:", searchResults);
        setResults(searchResults || []);
      } catch (error) {
        console.error("Search error:", error);
        setResults([]);
      } finally {
        setIsLoading(false);
      }
    };

    const timeoutId = setTimeout(search, 300);
    return () => clearTimeout(timeoutId);
  }, [query]);

  if (!isOpen) return null;

  // Helper functions
  const getResultIcon = (type) => {
    switch (type) {
      case "blog":
        return <FileText className="w-4 h-4 text-gray-500" />;
      case "product":
        return <Box className="w-4 h-4 text-blue-500" />;
      case "category":
        return <Tag className="w-4 h-4 text-green-500" />;
      default:
        return <SearchIcon className="w-4 h-4 text-gray-400" />;
    }
  };

  const formatResultTitle = (result) =>
    result?.title || result?.name || "Untitled";

  const formatResultSubtitle = (result) =>
    result?.excerpt ||
    result?.description ||
    (result?.price ? `$${result.price}` : "");

  const getResultUrl = (result) => {
    if (!result) return "#";
    if (result.url) return result.url;
    if (result.type === "blog" && result.slug) return `/blog/${result.slug}`;
    if (result.type === "product" && result.slug)
      return `/product/${result.slug}`;
    if (result.type === "category" && result.slug)
      return `/products/${result.slug}`;
    return "#";
  };

  const handleLinkClick = (e, url) => {
    // Don't interfere with cmd/ctrl click (open in new tab)
    if (e.metaKey || e.ctrlKey || e.shiftKey) {
      return;
    }

    // Let the Link handle the navigation naturally
    // Don't prevent default or close immediately
  };

  return (
    <div
      ref={searchRef}
      data-search-preview
      className="absolute top-full left-0 right-0 mt-2 bg-white rounded-xl shadow-2xl border border-gray-100 overflow-hidden z-50"
    >
      <div className="max-h-[60vh] overflow-y-auto">
        {isLoading ? (
          <div className="p-4 text-center text-gray-500">
            <div className="flex flex-col items-center space-y-4">
              <Loader2 className="animate-spin h-8 w-8 text-blue-500" />
              <p>Searching...</p>
            </div>
          </div>
        ) : results.length > 0 ? (
          <>
            <div className="sticky top-0 bg-white border-b border-gray-100 px-4 py-2.5 flex justify-between items-center">
              <h3 className="text-sm font-medium text-gray-500">
                Search Results
              </h3>
              <button
                onClick={onClose}
                className="text-gray-400 hover:text-gray-600 transition-colors"
                aria-label="Close search"
              >
                <X className="w-5 h-5" />
              </button>
            </div>
            <ul className="divide-y divide-gray-100">
              {results.map((result) => (
                <li key={`${result.type}-${result.id}`}>
                  <Link
                    href={getResultUrl(result)}
                    className="flex items-center p-4 hover:bg-gray-50 transition-colors group"
                    onClick={(e) => handleLinkClick(e, getResultUrl(result))}
                  >
                    <div className="shrink-0 mr-3">
                      {result.type === "product" && result.image ? (
                        <div className="w-10 h-10 relative rounded-md overflow-hidden bg-gray-100">
                          <Image
                            src={result.image}
                            alt={result.name}
                            fill
                            className="object-cover"
                            sizes="40px"
                          />
                        </div>
                      ) : (
                        <div className="w-10 h-10 rounded-md bg-gray-50 flex items-center justify-center">
                          {getResultIcon(result.type)}
                        </div>
                      )}
                    </div>
                    <div className="flex-1 min-w-0">
                      <p className="text-sm font-medium text-gray-900 truncate group-hover:text-primary">
                        {formatResultTitle(result)}
                      </p>
                      <p className="text-xs text-gray-500 truncate">
                        {formatResultSubtitle(result)}
                      </p>
                    </div>
                    <div className="ml-2">
                      <span
                        className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                          result.type === "blog"
                            ? "bg-purple-100 text-purple-800"
                            : result.type === "product"
                            ? "bg-blue-100 text-blue-800"
                            : "bg-green-100 text-green-800"
                        }`}
                      >
                        {result.type.charAt(0).toUpperCase() +
                          result.type.slice(1)}
                      </span>
                    </div>
                  </Link>
                </li>
              ))}
            </ul>
            <div className="sticky bottom-0 bg-linear-to-t from-white to-white/80 border-t border-gray-100 px-4 py-2.5 text-center">
              <Link
                href={`/search?q=${encodeURIComponent(query)}`}
                className="text-sm font-medium text-primary hover:underline"
                onClick={(e) =>
                  handleLinkClick(e, `/search?q=${encodeURIComponent(query)}`)
                }
              >
                View all results for "{query}"
              </Link>
            </div>
          </>
        ) : query ? (
          <div className="p-6 text-center">
            <SearchIcon className="w-8 h-8 mx-auto text-gray-300 mb-2" />
            <p className="text-gray-500">No results found for "{query}"</p>
            <p className="text-sm text-gray-400 mt-1">
              Try different keywords or check your spelling
            </p>
          </div>
        ) : (
          <div className="p-6 text-center">
            <SearchIcon className="w-8 h-8 mx-auto text-gray-300 mb-2" />
            <p className="text-gray-500">
              Type to search products, articles, and more
            </p>
          </div>
        )}
      </div>
    </div>
  );
}
