"use client";

import { Loader2 } from "lucide-react";
import { useEffect, useState } from "react";
import Image from "next/image";
import { motion, AnimatePresence } from "framer-motion";

export default function LoadingOverlay({ children }) {
  const [isLoading, setIsLoading] = useState(true);
  const [contentReady, setContentReady] = useState(false);

  useEffect(() => {
    // Minimum loading time to prevent flash
    const minLoadTime = 400;
    const startTime = Date.now();

    // Mark content as ready immediately so it can render in background
    setContentReady(true);

    const handleLoadComplete = () => {
      const elapsed = Date.now() - startTime;
      const remainingTime = Math.max(0, minLoadTime - elapsed);

      setTimeout(() => {
        setIsLoading(false);
      }, remainingTime);
    };

    // Wait for page to be fully loaded
    if (document.readyState === "complete") {
      handleLoadComplete();
    } else {
      window.addEventListener("load", handleLoadComplete);
      // Also set a maximum timeout to ensure loading screen doesn't stay forever
      const maxTimeout = setTimeout(() => {
        setIsLoading(false);
      }, 3000);

      return () => {
        window.removeEventListener("load", handleLoadComplete);
        clearTimeout(maxTimeout);
      };
    }
  }, []);

  return (
    <>
      {/* Content - loads in background */}
      <motion.div
        initial={{ opacity: 0 }}
        animate={{ opacity: contentReady && !isLoading ? 1 : 0 }}
        transition={{ duration: 0.6, ease: "easeInOut" }}
      >
        {children}
      </motion.div>

      {/* Loading Overlay - fades out smoothly */}
      <AnimatePresence>
        {isLoading && (
          <motion.div
            initial={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            transition={{ duration: 0.6, ease: "easeInOut" }}
            className="fixed inset-0 z-50 flex items-center justify-center bg-linear-to-br from-gray-50 via-white to-gray-50"
            style={{ fontFamily: "Inter, sans-serif" }}
          >
            <div className="glass rounded-2xl p-8 shadow-2xl backdrop-blur-md bg-white/80 border border-white/20 max-w-sm w-full mx-4">
              <div className="flex flex-col items-center space-y-6">
                {/* Logo */}
                <div className="relative w-16 h-16 animate-pulse">
                  <Image
                    src="/logo.png"
                    alt="Intelligent Technology"
                    fill
                    className="object-contain"
                    priority
                  />
                </div>

                {/* Spinner */}
                <div className="relative">
                  <Loader2
                    className="animate-spin h-8 w-8"
                    style={{ color: "#013e65" }}
                  />
                  <div
                    className="absolute inset-0 rounded-full border-2 border-transparent border-t-current animate-spin"
                    style={{ color: "#013e65", animationDuration: "1.5s" }}
                  ></div>
                </div>

                {/* Text */}
                <div className="text-center space-y-2">
                  <h2 className="text-lg font-semibold text-gray-800">
                    Intelligent Technology
                  </h2>
                  <p className="text-sm text-gray-600 animate-pulse">
                    Loading...
                  </p>
                </div>

                {/* Progress dots */}
                <div className="flex space-x-1">
                  <div
                    className="w-2 h-2 bg-[#013e65] rounded-full animate-bounce"
                    style={{ animationDelay: "0ms" }}
                  ></div>
                  <div
                    className="w-2 h-2 bg-[#013e65] rounded-full animate-bounce"
                    style={{ animationDelay: "150ms" }}
                  ></div>
                  <div
                    className="w-2 h-2 bg-[#013e65] rounded-full animate-bounce"
                    style={{ animationDelay: "300ms" }}
                  ></div>
                </div>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </>
  );
}
