'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { ChevronLeft, ChevronRight, Sparkles, ArrowRight, Zap } from 'lucide-react';
import {heroSlides} from '../data/staticData';

export default function Hero() {
  const [currentSlide, setCurrentSlide] = useState(0);
  const [isAutoPlaying, setIsAutoPlaying] = useState(true);
  const [direction, setDirection] = useState('next');

  useEffect(() => {
    if (!isAutoPlaying) return;

    const interval = setInterval(() => {
      setDirection('next');
      setCurrentSlide((prev) => (prev + 1) % heroSlides.length);
    }, 6000);

    return () => clearInterval(interval);
  }, [isAutoPlaying]);

  const nextSlide = () => {
    setDirection('next');
    setCurrentSlide((prev) => (prev + 1) % heroSlides.length);
    setIsAutoPlaying(false);
  };

  const prevSlide = () => {
    setDirection('prev');
    setCurrentSlide((prev) => (prev - 1 + heroSlides.length) % heroSlides.length);
    setIsAutoPlaying(false);
  };

  const goToSlide = (index) => {
    setDirection(index > currentSlide ? 'next' : 'prev');
    setCurrentSlide(index);
    setIsAutoPlaying(false);
  };

  return (
    <div className="relative w-full h-[500px] sm:h-[600px] md:h-[650px] lg:h-[750px] overflow-hidden bg-gradient-to-br from-gray-900 via-primary-dark to-black">
      {/* Animated Background Pattern */}
      <div className="absolute inset-0 opacity-10">
        <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjAiIGhlaWdodD0iNjAiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PGRlZnM+PHBhdHRlcm4gaWQ9ImdyaWQiIHdpZHRoPSI2MCIgaGVpZ2h0PSI2MCIgcGF0dGVyblVuaXRzPSJ1c2VyU3BhY2VPblVzZSI+PHBhdGggZD0iTSAxMCAwIEwgMTAgNjAgTSAwIDEwIEwgNjAgMTAgTSAyMCAwIEwgMjAgNjAgTSAwIDIwIEwgNjAgMjAgTSAzMCAwIEwgMzAgNjAgTSAwIDMwIEwgNjAgMzAgTSA0MCAwIEwgNDAgNjAgTSAwIDQwIEwgNjAgNDAgTSA1MCAwIEwgNTAgNjAgTSAwIDUwIEwgNjAgNTAiIGZpbGw9Im5vbmUiIHN0cm9rZT0id2hpdGUiIHN0cm9rZS13aWR0aD0iMSIvPjwvcGF0dGVybj48L2RlZnM+PHJlY3Qgd2lkdGg9IjEwMCUiIGhlaWdodD0iMTAwJSIgZmlsbD0idXJsKCNncmlkKSIvPjwvc3ZnPg==')] animate-pulse"></div>
      </div>

      {/* Slides */}
      {heroSlides.map((slide, index) => (
        <div
          key={slide.id}
          className={`absolute inset-0 transition-all duration-1000 ${
            index === currentSlide ? 'opacity-100 scale-100' : 'opacity-0 scale-105'
          }`}
        >
          {/* Background Image with Parallax Effect */}
          <div
            className="absolute inset-0 bg-cover bg-center transition-transform duration-1000"
            style={{ 
              backgroundImage: `url(${slide.image})`,
              transform: index === currentSlide ? 'scale(1)' : 'scale(1.1)'
            }}
          >
            <div className={`absolute inset-0 bg-gradient-to-r ${slide.gradient}`} />
            <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent" />
          </div>

          {/* Content with Glass Effect */}
          <div className="relative h-full container mx-auto px-4 sm:px-6 flex items-center">
            <div className="max-w-3xl w-full">
              {/* Subtitle Badge */}
              <div className="inline-flex items-center gap-1.5 md:gap-2 glass-white px-3 py-1.5 md:px-4 md:py-2 rounded-full mb-4 md:mb-6 animate-in fade-in slide-in-from-left-4 duration-700">
                <Sparkles size={14} className="text-secondary animate-pulse md:w-4 md:h-4" />
                <span className="text-xs md:text-sm font-bold text-primary" style={{ fontFamily: 'Poppins, sans-serif' }}>
                  {slide.subtitle}
                </span>
              </div>

              {/* Main Title with Gradient */}
              <h1 
                className="text-3xl sm:text-4xl md:text-5xl lg:text-6xl xl:text-7xl font-black mb-4 md:mb-6 leading-tight animate-in fade-in slide-in-from-left-8 duration-700 delay-100"
                style={{ fontFamily: 'Poppins, sans-serif' }}
              >
                <span className="text-white drop-shadow-2xl">
                  {slide.title}
                </span>
              </h1>

              {/* Description with Glass Card */}
              <div className="glass-dark rounded-xl md:rounded-2xl p-4 md:p-6 mb-6 md:mb-8 max-w-2xl animate-in fade-in slide-in-from-left-12 duration-700 delay-200">
                <p className="text-sm sm:text-base md:text-lg lg:text-xl text-gray-100 leading-relaxed" style={{ fontFamily: 'Inter, sans-serif' }}>
                  {slide.description}
                </p>
              </div>

              {/* CTA Buttons */}
              <div className="flex flex-col sm:flex-row gap-3 md:gap-4 animate-in fade-in slide-in-from-left-16 duration-700 delay-300">
                <Link
                  href={slide.buttonLink}
                  className="group relative inline-flex items-center justify-center gap-2 md:gap-3 bg-gradient-to-r from-secondary to-secondary-light text-white px-6 py-3 md:px-8 md:py-4 rounded-xl font-bold text-base md:text-lg hover:shadow-2xl hover:shadow-secondary/50 transition-all hover:scale-105 overflow-hidden"
                  style={{ fontFamily: 'Poppins, sans-serif' }}
                >
                  <div className="absolute inset-0 bg-white/20 translate-x-full group-hover:translate-x-0 transition-transform duration-500"></div>
                  <span className="relative">{slide.buttonText}</span>
                  <ArrowRight size={18} className="relative group-hover:translate-x-1 transition-transform md:w-5 md:h-5" />
                </Link>
                <Link
                  href="/contact"
                  className="group inline-flex items-center justify-center gap-2 md:gap-3 glass-white text-primary px-6 py-3 md:px-8 md:py-4 rounded-xl font-bold text-base md:text-lg hover:shadow-xl transition-all hover:scale-105 border-2 border-white/30"
                  style={{ fontFamily: 'Poppins, sans-serif' }}
                >
                  <Zap size={18} className="group-hover:text-secondary transition-colors md:w-5 md:h-5" />
                  <span>Get Started</span>
                </Link>
              </div>

              {/* Stats Bar */}
              <div className="flex flex-wrap gap-3 md:gap-4 lg:gap-6 mt-6 md:mt-8 lg:mt-12 animate-in fade-in duration-700 delay-500">
                <div className="glass-dark px-4 py-2 md:px-6 md:py-3 rounded-lg md:rounded-xl">
                  <div className="text-xl md:text-2xl lg:text-3xl font-bold text-white" style={{ fontFamily: 'Poppins, sans-serif' }}>50+</div>
                  <div className="text-xs md:text-sm text-gray-300" style={{ fontFamily: 'Inter, sans-serif' }}>Projects</div>
                </div>
                <div className="glass-dark px-4 py-2 md:px-6 md:py-3 rounded-lg md:rounded-xl">
                  <div className="text-xl md:text-2xl lg:text-3xl font-bold text-white" style={{ fontFamily: 'Poppins, sans-serif' }}>5+</div>
                  <div className="text-xs md:text-sm text-gray-300" style={{ fontFamily: 'Inter, sans-serif' }}>Years</div>
                </div>
                <div className="glass-dark px-4 py-2 md:px-6 md:py-3 rounded-lg md:rounded-xl">
                  <div className="text-xl md:text-2xl lg:text-3xl font-bold text-white" style={{ fontFamily: 'Poppins, sans-serif' }}>24/7</div>
                  <div className="text-xs md:text-sm text-gray-300" style={{ fontFamily: 'Inter, sans-serif' }}>Support</div>
                </div>
              </div>
            </div>
          </div>
        </div>
      ))}

      {/* Modern Navigation Arrows */}
      <button
        onClick={prevSlide}
        className="absolute left-2 sm:left-4 md:left-6 top-1/2 -translate-y-1/2 glass-white hover:glass p-2 md:p-3 lg:p-4 rounded-xl md:rounded-2xl transition-all z-10 hover:scale-110 group"
        aria-label="Previous slide"
      >
        <ChevronLeft size={20} className="text-primary group-hover:text-secondary transition-colors md:w-6 md:h-6 lg:w-7 lg:h-7" />
      </button>
      <button
        onClick={nextSlide}
        className="absolute right-2 sm:right-4 md:right-6 top-1/2 -translate-y-1/2 glass-white hover:glass p-2 md:p-3 lg:p-4 rounded-xl md:rounded-2xl transition-all z-10 hover:scale-110 group"
        aria-label="Next slide"
      >
        <ChevronRight size={20} className="text-primary group-hover:text-secondary transition-colors md:w-6 md:h-6 lg:w-7 lg:h-7" />
      </button>

      {/* Modern Progress Indicators */}
      <div className="absolute bottom-4 md:bottom-6 lg:bottom-8 left-1/2 -translate-x-1/2 flex gap-2 md:gap-3 z-10">
        {heroSlides.map((_, index) => (
          <button
            key={index}
            onClick={() => goToSlide(index)}
            className="group relative"
            aria-label={`Go to slide ${index + 1}`}
          >
            <div className={`h-1.5 md:h-2 rounded-full transition-all duration-500 ${
              index === currentSlide
                ? 'w-8 md:w-12 bg-gradient-to-r from-secondary to-secondary-light'
                : 'w-1.5 md:w-2 bg-white/50 hover:bg-white/70'
            }`}>
              {index === currentSlide && (
                <div className="absolute inset-0 bg-white/50 rounded-full animate-pulse"></div>
              )}
            </div>
          </button>
        ))}
      </div>

      {/* Floating Elements */}
      <div className="absolute top-20 right-20 w-20 h-20 bg-secondary/20 rounded-full blur-3xl animate-float"></div>
      <div className="absolute bottom-40 left-40 w-32 h-32 bg-primary/20 rounded-full blur-3xl animate-float" style={{ animationDelay: '2s' }}></div>
    </div>
  );
}
