"use client";

import { useState, useEffect, useRef } from "react";
import Link from "next/link";
import { usePathname, useRouter } from "next/navigation";
import Image from "next/image";
import {
  Search,
  Phone,
  Mail,
  MapPin,
  ChevronDown,
  Menu,
  X,
  Sparkles,
} from "lucide-react";
import { navigationLinks, contactInfo } from "@/lib/navigation";
import SearchPreview from "./SearchPreview";

export default function HeaderV2() {
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [openDropdown, setOpenDropdown] = useState(null);
  const [searchQuery, setSearchQuery] = useState("");
  const [isSearchFocused, setIsSearchFocused] = useState(false);
  const [scrolled, setScrolled] = useState(false);
  const pathname = usePathname();
  const router = useRouter();
  const searchInputRef = useRef(null);
  const searchContainerRef = useRef(null);

  const handleSearch = (e) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      router.push(`/search?q=${encodeURIComponent(searchQuery)}`);
      setSearchQuery("");
      setIsSearchFocused(false);
    }
  };

  // Close search when clicking outside
  useEffect(() => {
    const handleClickOutside = (event) => {
      // Check if clicking inside search container (desktop)
      if (
        searchContainerRef.current &&
        searchContainerRef.current.contains(event.target)
      ) {
        return; // Don't close if clicking inside desktop search container
      }

      // Check if click is inside any SearchPreview component (desktop or mobile)
      const searchPreviews = document.querySelectorAll("[data-search-preview]");
      for (const preview of searchPreviews) {
        if (preview.contains(event.target)) {
          return; // Don't close if clicking inside any SearchPreview
        }
      }

      // Check if clicking on mobile overlay background
      const mobileOverlay = event.target.closest(
        ".fixed.inset-0.bg-black\\/50"
      );
      if (mobileOverlay && event.target !== mobileOverlay) {
        return; // Don't close if clicking on mobile search content
      }

      // Close search if clicking outside
      setIsSearchFocused(false);
    };

    if (isSearchFocused) {
      // Add delay to prevent immediate closing when opening
      const timeoutId = setTimeout(() => {
        document.addEventListener("mousedown", handleClickOutside);
      }, 150);

      return () => {
        clearTimeout(timeoutId);
        document.removeEventListener("mousedown", handleClickOutside);
      };
    }
  }, [isSearchFocused]);

  // Close search when route changes
  useEffect(() => {
    setIsSearchFocused(false);
  }, [pathname]);

  // Handle keyboard navigation
  useEffect(() => {
    const handleKeyDown = (e) => {
      // Close search on Escape key
      if (e.key === "Escape" && isSearchFocused) {
        setIsSearchFocused(false);
      }
      // Focus search on Cmd+K / Ctrl+K
      if ((e.metaKey || e.ctrlKey) && e.key === "k") {
        e.preventDefault();
        searchInputRef.current?.focus();
      }
    };

    document.addEventListener("keydown", handleKeyDown);
    return () => document.removeEventListener("keydown", handleKeyDown);
  }, [isSearchFocused]);

  // Helper function to check if a link is active
  const isActive = (href) => {
    if (href === "/") {
      return pathname === href;
    }
    return pathname.startsWith(href) && href !== "/";
  };

  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 20);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  return (
    <header
      className={`w-full fixed top-0 z-[100] transition-all duration-300 ${
        scrolled
          ? "glass-white shadow-2xl backdrop-blur-xl border-b border-gray-200/30"
          : "bg-white/95 backdrop-blur-md"
      }`}
    >
      {/* Top Bar with Gradient */}
      <div className="bg-gradient-to-r from-primary via-primary-light to-secondary text-white relative overflow-hidden">
        <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAiIGhlaWdodD0iNDAiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PGRlZnM+PHBhdHRlcm4gaWQ9ImdyaWQiIHdpZHRoPSI0MCIgaGVpZ2h0PSI0MCIgcGF0dGVyblVuaXRzPSJ1c2VyU3BhY2VPblVzZSI+PHBhdGggZD0iTSAwIDEwIEwgNDAgMTAgTSAxMCAwIEwgMTAgNDAgTSAwIDIwIEwgNDAgMjAgTSAyMCAwIEwgMjAgNDAgTSAwIDMwIEwgNDAgMzAgTSAzMCAwIEwgMzAgNDAiIGZpbGw9Im5vbmUiIHN0cm9rZT0id2hpdGUiIHN0cm9rZS1vcGFjaXR5PSIwLjAzIiBzdHJva2Utd2lkdGg9IjEiLz48L3BhdHRlcm4+PC9kZWZzPjxyZWN0IHdpZHRoPSIxMDAlIiBoZWlnaHQ9IjEwMCUiIGZpbGw9InVybCgjZ3JpZCkiLz48L3N2Zz4=')] opacity-30"></div>
        <div className="container mx-auto px-4 py-2 md:py-2.5 relative">
          <div className="flex flex-wrap justify-between items-center text-xs md:text-sm gap-2">
            <div className="flex flex-wrap gap-2 md:gap-4 lg:gap-6">
              <a
                href={`tel:${contactInfo.phone}`}
                className="flex items-center gap-1.5 md:gap-2 hover:text-white/80 transition-all hover:scale-105"
              >
                <div className="bg-white/20 p-1 md:p-1.5 rounded-full">
                  <Phone size={12} className="md:hidden" />
                  <Phone size={14} className="hidden md:block" />
                </div>
                <span className="font-medium text-xs md:text-sm">
                  {contactInfo.phone}
                </span>
              </a>
              <a
                href={`mailto:${contactInfo.email}`}
                className="flex items-center gap-1.5 md:gap-2 hover:text-white/80 transition-all hover:scale-105"
              >
                <div className="bg-white/20 p-1 md:p-1.5 rounded-full">
                  <Mail size={12} className="md:hidden" />
                  <Mail size={14} className="hidden md:block" />
                </div>
                <span className="sm:inline font-medium text-xs md:text-sm">
                  {contactInfo.email}
                </span>
              </a>
              <div className="hidden md:flex items-center gap-2">
                <div className="bg-white/20 p-1.5 rounded-full">
                  <MapPin size={12} className="md:hidden" />
                  <MapPin size={14} className="hidden md:block" />
                </div>
                <span className="hidden lg:inline font-medium">
                  {contactInfo.address}
                </span>
              </div>
            </div>
            <div className="hidden md:flex items-center gap-2 text-xs font-semibold">
              <Sparkles size={14} className="animate-pulse" />
              <span className="hidden lg:inline">
                Premium Industrial Solutions
              </span>
            </div>
          </div>
        </div>
      </div>

      {/* Main Header */}
      <div className="container mx-auto px-4 py-3 md:py-2">
        <div className="flex items-center justify-between gap-2 md:gap-4">
          {/* Logo */}
          <Link href="/" className="flex-shrink-0 group">
            <div className="relative">
              <Image
                src="/logo-transparent.png"
                alt="Intellegent Technology"
                width={180}
                height={60}
                className="h-10 md:h-12 lg:h-14 w-auto object-contain transition-all group-hover:scale-105"
                priority
              />
              <div className="absolute -inset-2 bg-gradient-to-r from-primary/20 to-secondary/20 rounded-lg blur-xl opacity-0 group-hover:opacity-100 transition-opacity -z-10"></div>
            </div>
          </Link>

          {/* Modern Search Bar - Desktop */}
          <div
            className="hidden lg:block flex-1 max-w-2xl mx-4 2xl:mx-8 relative"
            ref={searchContainerRef}
          >
            <form onSubmit={handleSearch} className="relative w-full">
              <div className="absolute -inset-0.5 bg-gradient-to-r from-primary to-secondary rounded-xl blur opacity-20 group-hover:opacity-40 transition duration-300"></div>
              <div className="relative flex items-center">
                <input
                  ref={searchInputRef}
                  type="text"
                  placeholder="Search products, solutions, insights..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  onFocus={() => setIsSearchFocused(true)}
                  className="w-full px-6 py-3.5 pr-14 bg-white border-2 border-gray-200 rounded-xl focus:outline-none focus:border-primary focus:ring-2 focus:ring-primary/20 transition-all font-['Inter'] text-sm"
                  style={{ fontFamily: "Inter, sans-serif" }}
                  aria-label="Search"
                />
                <div className="absolute inset-y-0 right-0 flex items-center pr-3">
                  <kbd className="hidden md:inline-flex items-center px-2 py-1.5 border border-gray-200 rounded text-xs text-gray-500 mr-2">
                    {navigator.userAgent.includes("Mac") ? "⌘K" : "Ctrl+K"}
                  </kbd>
                  <button
                    type="submit"
                    className="bg-gradient-to-r from-primary to-primary-light text-white p-2 rounded-lg hover:shadow-lg transition-all hover:scale-105"
                    aria-label="Search"
                  >
                    <Search size={18} />
                  </button>
                </div>
              </div>
              {isSearchFocused && (
                <div className="absolute top-full left-0 right-0 mt-2 z-[200]">
                  <SearchPreview
                    query={searchQuery}
                    onClose={() => setIsSearchFocused(false)}
                    isOpen={isSearchFocused}
                  />
                </div>
              )}
            </form>
          </div>

          {/* Modern CTA Button - Desktop */}
          <Link
            href="/contact"
            className="hidden lg:flex items-center gap-2 bg-gradient-to-r from-secondary to-secondary-light text-white px-4 lg:px-6 py-2.5 lg:py-3 rounded-xl font-bold hover:shadow-xl transition-all hover:scale-105 whitespace-nowrap relative overflow-hidden group text-sm lg:text-base"
            style={{ fontFamily: "Poppins, sans-serif" }}
          >
            <div className="absolute inset-0 bg-white/20 translate-y-full group-hover:translate-y-0 transition-transform duration-300"></div>
            <span className="relative">Get Quote</span>
            <Sparkles size={16} className="relative lg:w-[18px] lg:h-[18px]" />
          </Link>

          <div className="flex items-center gap-2 flex-row">
            {" "}
            {/* Mobile Search Button */}
            <div className="lg:hidden relative">
              <button
                className="p-2 text-gray-600 hover:text-primary transition-colors"
                onClick={() => {
                  setIsSearchFocused(!isSearchFocused);
                  if (!isSearchFocused) {
                    setTimeout(() => searchInputRef.current?.focus(), 100);
                  }
                }}
                aria-label="Toggle search"
              >
                <Search size={20} />
              </button>

              {/* Mobile Search Input */}
              {isSearchFocused && (
                <div
                  className="fixed inset-0 bg-black/50 z-[150]"
                  onClick={(e) => {
                    // Only close if clicking directly on the overlay, not on children
                    if (e.target === e.currentTarget) {
                      setIsSearchFocused(false);
                    }
                  }}
                >
                  <div
                    className="absolute top-20 left-4 right-4 sm:left-1/2 sm:-translate-x-1/2 w-auto sm:w-[calc(100%-2rem)] sm:max-w-lg"
                    data-search-preview
                  >
                    <form
                      onSubmit={handleSearch}
                      className="relative w-full mb-4"
                    >
                      <input
                        ref={searchInputRef}
                        type="text"
                        placeholder="Search..."
                        value={searchQuery}
                        onChange={(e) => setSearchQuery(e.target.value)}
                        className="w-full px-4 py-3 pr-12 bg-white border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-transparent text-base shadow-lg"
                        autoFocus
                      />
                      <button
                        type="submit"
                        className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-primary"
                      >
                        <Search size={20} />
                      </button>
                    </form>
                    {searchQuery && (
                      <div className="bg-white rounded-xl shadow-xl max-h-[60vh] overflow-y-auto">
                        <SearchPreview
                          query={searchQuery}
                          onClose={() => setIsSearchFocused(false)}
                          isOpen={isSearchFocused}
                        />
                      </div>
                    )}
                  </div>
                </div>
              )}
            </div>
            {/* Mobile Menu Button */}
            <button
              onClick={() => setIsMenuOpen(!isMenuOpen)}
              className="lg:hidden p-2 text-primary hover:bg-primary/10 rounded-xl transition-all"
              aria-label="Toggle menu"
            >
              {isMenuOpen ? <X size={24} /> : <Menu size={24} />}
            </button>{" "}
          </div>
        </div>
      </div>

      {/* Modern Navigation */}
      <nav className="bg-gradient-to-r from-gray-900 via-primary-dark to-gray-900 relative z-10">
        <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjAiIGhlaWdodD0iNjAiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PGRlZnM+PHBhdHRlcm4gaWQ9ImRvdHMiIHdpZHRoPSI2MCIgaGVpZ2h0PSI2MCIgcGF0dGVyblVuaXRzPSJ1c2VyU3BhY2VPblVzZSI+PGNpcmNsZSBjeD0iMiIgY3k9IjIiIHI9IjEiIGZpbGw9IndoaXRlIiBmaWxsLW9wYWNpdHk9IjAuMSIvPjwvcGF0dGVybj48L2RlZnM+PHJlY3Qgd2lkdGg9IjEwMCUiIGhlaWdodD0iMTAwJSIgZmlsbD0idXJsKCNkb3RzKSIvPjwvc3ZnPg==')] opacity-50"></div>
        <div className="container mx-auto px-4 relative">
          {/* Desktop Navigation */}
          <ul
            className="hidden lg:flex items-center justify-center gap-2"
            style={{ fontFamily: "Poppins, sans-serif" }}
          >
            {navigationLinks.map((link) => (
              <li
                key={link.label}
                className="relative group"
                onMouseEnter={() =>
                  link.dropdown && setOpenDropdown(link.label)
                }
                onMouseLeave={() => setOpenDropdown(null)}
                onClick={() =>
                  link.dropdown &&
                  setOpenDropdown(
                    openDropdown === link.label ? null : link.label
                  )
                }
              >
                <div className="flex items-center">
                  <Link
                    href={link.href}
                    className={`flex items-center gap-1 px-3 md:px-4 lg:px-5 py-3 md:py-4 ${
                      isActive(link.href)
                        ? "text-secondary-light"
                        : "text-white hover:text-secondary-light"
                    } transition-all font-semibold text-xs md:text-sm relative overflow-hidden group`}
                    style={{
                      textShadow: isActive(link.href)
                        ? "0 0 8px rgba(99, 102, 241, 0.5)"
                        : "none",
                    }}
                    onClick={(e) => {
                      if (link.dropdown) {
                        e.preventDefault();
                        setOpenDropdown(
                          openDropdown === link.label ? null : link.label
                        );
                      }
                    }}
                  >
                    <span className="absolute inset-0 bg-white/10 translate-y-full group-hover:translate-y-0 transition-transform duration-300"></span>
                    <span className="relative">{link.label}</span>
                    {link.dropdown && (
                      <ChevronDown
                        size={16}
                        className={`relative transition-transform duration-300 ${
                          openDropdown === link.label ? "rotate-180" : ""
                        }`}
                      />
                    )}
                  </Link>
                </div>

                {/* Modern Dropdown Menu */}
                {link.dropdown && openDropdown === link.label && (
                  <div className="absolute top-full left-0 pt-2 z-[1000] min-w-[240px]">
                    <ul className="bg-white rounded-2xl shadow-2xl border border-gray-100 py-2">
                      {link.dropdown.map((item) => (
                        <li key={item.label}>
                          <Link
                            href={item.href}
                            className={`block px-6 py-3 ${
                              pathname === item.href
                                ? "text-primary bg-primary/5"
                                : "text-gray-700 hover:text-primary hover:bg-primary/5"
                            } transition-all font-medium text-sm relative overflow-hidden group`}
                            onClick={() => setOpenDropdown(null)}
                          >
                            <span className="absolute left-0 top-1/2 -translate-y-1/2 w-1 h-0 bg-gradient-to-b from-primary to-secondary group-hover:h-full transition-all duration-300 rounded-r"></span>
                            <span className="relative pl-2">{item.label}</span>
                          </Link>
                        </li>
                      ))}
                    </ul>
                  </div>
                )}
              </li>
            ))}
          </ul>

          {/* Mobile Navigation */}
          {isMenuOpen && (
            <div className="lg:hidden py-4">
              <ul className="space-y-2">
                {navigationLinks.map((link) => (
                  <li key={link.label} className="space-y-2">
                    <div className="flex items-center justify-between">
                      <Link
                        href={link.href}
                        className={`block px-4 py-3 ${
                          isActive(link.href)
                            ? "text-primary bg-primary/10"
                            : "text-white hover:bg-white/10"
                        } rounded-lg transition-all font-medium w-full`}
                        onClick={() => !link.dropdown && setIsMenuOpen(false)}
                      >
                        <div className="flex items-center">
                          <span className="relative">{link.label}</span>
                          {isActive(link.href) && (
                            <span className="ml-2 w-2 h-2 bg-primary rounded-full"></span>
                          )}
                        </div>
                      </Link>
                      {link.dropdown && (
                        <button
                          className="p-3 text-white hover:bg-gray-100 rounded-lg transition-all"
                          onClick={(e) => {
                            e.preventDefault();
                            e.stopPropagation();
                            setOpenDropdown(
                              openDropdown === link.label ? null : link.label
                            );
                          }}
                          aria-expanded={openDropdown === link.label}
                          aria-label={`Toggle ${link.label} submenu`}
                        >
                          <ChevronDown
                            size={18}
                            className={`transition-transform duration-300 ${
                              openDropdown === link.label ? "rotate-180" : ""
                            }`}
                          />
                        </button>
                      )}
                    </div>
                    {link.dropdown && openDropdown === link.label && (
                      <div className="pl-4 border-l-2 border-gray-100 ml-2">
                        <ul className="space-y-1">
                          {link.dropdown.map((item) => (
                            <li key={item.label}>
                              <Link
                                href={item.href}
                                className={`block px-4 py-2 ${
                                  pathname === item.href
                                    ? "text-primary bg-primary/5"
                                    : "text-white hover:bg-gray-50"
                                } rounded-lg transition-all text-sm`}
                                onClick={() => {
                                  setIsMenuOpen(false);
                                  setOpenDropdown(null);
                                }}
                              >
                                <div className="flex items-center">
                                  <span className="relative">{item.label}</span>
                                  {pathname === item.href && (
                                    <span className="ml-2 w-1.5 h-1.5 bg-primary rounded-full"></span>
                                  )}
                                </div>
                              </Link>
                            </li>
                          ))}
                        </ul>
                      </div>
                    )}
                  </li>
                ))}
              </ul>

              {/* Mobile CTA Button */}
              <Link
                href="/contact"
                className="flex items-center justify-center gap-2 mt-4 bg-gradient-to-r from-secondary to-secondary-light text-white px-6 py-3 rounded-xl font-bold hover:shadow-xl transition-all"
                onClick={() => setIsMenuOpen(false)}
                style={{ fontFamily: "Poppins, sans-serif" }}
              >
                Get Quote <Sparkles size={18} />
              </Link>
            </div>
          )}
        </div>
      </nav>
    </header>
  );
}
