import Link from 'next/link';
import Image from 'next/image';
import { Phone, Mail, MapPin, Send, ArrowRight, Sparkles } from 'lucide-react';
import { contactInfo, socialLinks } from '@/lib/navigation';

export default function FooterV2() {
  const currentYear = new Date().getFullYear();

  return (
    <footer className="relative bg-gradient-to-br from-gray-900 via-primary-dark to-black text-gray-300 overflow-hidden">
      {/* Animated Background Pattern */}
      <div className="absolute inset-0 opacity-5">
        <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjAiIGhlaWdodD0iNjAiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PGRlZnM+PHBhdHRlcm4gaWQ9ImdyaWQiIHdpZHRoPSI2MCIgaGVpZ2h0PSI2MCIgcGF0dGVyblVuaXRzPSJ1c2VyU3BhY2VPblVzZSI+PHBhdGggZD0iTSAxMCAwIEwgMTAgNjAgTSAwIDEwIEwgNjAgMTAgTSAyMCAwIEwgMjAgNjAgTSAwIDIwIEwgNjAgMjAgTSAzMCAwIEwgMzAgNjAgTSAwIDMwIEwgNjAgMzAgTSA0MCAwIEwgNDAgNjAgTSAwIDQwIEwgNjAgNDAgTSA1MCAwIEwgNTAgNjAgTSAwIDUwIEwgNjAgNTAiIGZpbGw9Im5vbmUiIHN0cm9rZT0id2hpdGUiIHN0cm9rZS13aWR0aD0iMSIvPjwvcGF0dGVybj48L2RlZnM+PHJlY3Qgd2lkdGg9IjEwMCUiIGhlaWdodD0iMTAwJSIgZmlsbD0idXJsKCNncmlkKSIvPjwvc3ZnPg==')]"></div>
      </div>

      {/* Floating Elements */}
      <div className="absolute top-20 right-20 w-40 h-40 bg-secondary/10 rounded-full blur-3xl"></div>
      <div className="absolute bottom-40 left-20 w-60 h-60 bg-primary/10 rounded-full blur-3xl"></div>

      {/* Main Footer */}
      <div className="container mx-auto px-4 py-16 relative">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-10">
          {/* Company Info */}
          <div className="lg:col-span-1">
            <div className="glass-dark p-4 rounded-2xl inline-block mb-6">
              <Image
                src="/logo.png"
                alt="Intellegent Technology"
                width={160}
                height={70}
                className="h-22 w-auto object-contain"
              />
            </div>
            <p className="text-sm mb-6 leading-relaxed text-gray-400" style={{ fontFamily: 'Inter, sans-serif' }}>
              Leading provider of industrial security solutions, ID card printers, and surveillance systems. 
              <span className="text-secondary font-semibold"> Trusted by businesses across Bangladesh.</span>
            </p>
            
            {/* Contact Info Cards */}
            <div className="space-y-3">
              <a href={`tel:${contactInfo.phone}`} className="group flex items-center gap-3 glass-dark p-3 rounded-xl hover:bg-white/10 transition-all">
                <div className="bg-gradient-to-br from-primary to-primary-light p-2 rounded-lg group-hover:scale-110 transition-transform">
                  <Phone size={16} className="text-white" />
                </div>
                <span className="text-sm font-medium group-hover:text-white transition-colors" style={{ fontFamily: 'Inter, sans-serif' }}>
                  {contactInfo.phone}
                </span>
              </a>
              <a href={`mailto:${contactInfo.email}`} className="group flex items-center gap-3 glass-dark p-3 rounded-xl hover:bg-white/10 transition-all">
                <div className="bg-gradient-to-br from-secondary to-secondary-light p-2 rounded-lg group-hover:scale-110 transition-transform">
                  <Mail size={16} className="text-white" />
                </div>
                <span className="text-sm font-medium group-hover:text-white transition-colors" style={{ fontFamily: 'Inter, sans-serif' }}>
                  {contactInfo.email}
                </span>
              </a>
              <div className="flex items-start gap-3 glass-dark p-3 rounded-xl">
                <div className="bg-gradient-to-br from-primary-dark to-primary p-2 rounded-lg">
                  <MapPin size={16} className="text-white" />
                </div>
                <span className="text-sm font-medium" style={{ fontFamily: 'Inter, sans-serif' }}>
                  {contactInfo.address}
                </span>
              </div>
            </div>
          </div>

          {/* About Links */}
          <div>
            <h3 className="text-white font-bold text-lg mb-6 flex items-center gap-2" style={{ fontFamily: 'Poppins, sans-serif' }}>
              <div className="w-1 h-6 bg-gradient-to-b from-primary to-secondary rounded-full"></div>
              About Us
            </h3>
            <ul className="space-y-3">
              {[
                { label: 'About Company', href: '/about' },
                { label: 'Our Team', href: '/about' },
                { label: 'Career Opportunities', href: '/about' },
                { label: 'News & Updates', href: '/blog' },
                { label: 'Our Projects', href: '/projects' },
              ].map((link) => (
                <li key={link.label}>
                  <Link 
                    href={link.href} 
                    className="group flex items-center gap-2 text-sm hover:text-secondary transition-all"
                    style={{ fontFamily: 'Inter, sans-serif' }}
                  >
                    <ArrowRight size={14} className="opacity-0 -translate-x-2 group-hover:opacity-100 group-hover:translate-x-0 transition-all text-secondary" />
                    <span className="group-hover:translate-x-1 transition-transform">{link.label}</span>
                  </Link>
                </li>
              ))}
            </ul>
          </div>

          {/* Quick Links */}
          <div>
            <h3 className="text-white font-bold text-lg mb-6 flex items-center gap-2" style={{ fontFamily: 'Poppins, sans-serif' }}>
              <div className="w-1 h-6 bg-gradient-to-b from-secondary to-secondary-light rounded-full"></div>
              Quick Links
            </h3>
            <ul className="space-y-3">
              {[
                { label: 'All Products', href: '/products' },
                { label: 'Contact Us', href: '/contact' },
                { label: 'Privacy Policy', href: '/privacy-policy' },
                { label: 'Terms & Conditions', href: '/terms' },
                { label: 'Sitemap', href: '/sitemap.xml' },
              ].map((link) => (
                <li key={link.label}>
                  <Link 
                    href={link.href} 
                    className="group flex items-center gap-2 text-sm hover:text-secondary transition-all"
                    style={{ fontFamily: 'Inter, sans-serif' }}
                  >
                    <ArrowRight size={14} className="opacity-0 -translate-x-2 group-hover:opacity-100 group-hover:translate-x-0 transition-all text-secondary" />
                    <span className="group-hover:translate-x-1 transition-transform">{link.label}</span>
                  </Link>
                </li>
              ))}
            </ul>
          </div>

          {/* Newsletter */}
          <div>
            <h3 className="text-white font-bold text-lg mb-6 flex items-center gap-2" style={{ fontFamily: 'Poppins, sans-serif' }}>
              <Sparkles size={20} className="text-secondary animate-pulse" />
              Stay Connected
            </h3>
            <p className="text-sm mb-6 text-gray-400" style={{ fontFamily: 'Inter, sans-serif' }}>
              Subscribe for exclusive updates, offers, and industry insights.
            </p>
            <form className="space-y-3">
              <div className="relative group">
                <input
                  type="text"
                  placeholder="Your Name"
                  className="w-full px-4 py-3 bg-white/5 border border-white/10 rounded-xl focus:outline-none focus:border-secondary focus:bg-white/10 text-sm transition-all text-white placeholder-gray-500"
                  style={{ fontFamily: 'Inter, sans-serif' }}
                />
              </div>
              <div className="relative group">
                <input
                  type="email"
                  placeholder="Your Email"
                  className="w-full px-4 py-3 bg-white/5 border border-white/10 rounded-xl focus:outline-none focus:border-secondary focus:bg-white/10 text-sm transition-all text-white placeholder-gray-500"
                  style={{ fontFamily: 'Inter, sans-serif' }}
                />
              </div>
              <div className="relative group">
                <textarea
                  placeholder="Your Message (Optional)"
                  rows="3"
                  className="w-full px-4 py-3 bg-white/5 border border-white/10 rounded-xl focus:outline-none focus:border-secondary focus:bg-white/10 text-sm resize-none transition-all text-white placeholder-gray-500"
                  style={{ fontFamily: 'Inter, sans-serif' }}
                ></textarea>
              </div>
              <button
                type="submit"
                className="group w-full bg-gradient-to-r from-secondary to-secondary-light text-white px-6 py-3 rounded-xl font-bold hover:shadow-xl hover:shadow-secondary/30 transition-all flex items-center justify-center gap-2 relative overflow-hidden"
                style={{ fontFamily: 'Poppins, sans-serif' }}
              >
                <div className="absolute inset-0 bg-white/20 translate-y-full group-hover:translate-y-0 transition-transform duration-300"></div>
                <Send size={18} className="relative" />
                <span className="relative">Send Message</span>
              </button>
            </form>
          </div>
        </div>
      </div>

      {/* Bottom Footer */}
      <div className="border-t border-white/10 relative">
        <div className="container mx-auto px-4 py-6">
          <div className="flex flex-col md:flex-row justify-between items-center gap-6">
            {/* Copyright */}
            <p className="text-sm text-center md:text-left text-gray-400" style={{ fontFamily: 'Inter, sans-serif' }}>
              © {currentYear} <span className="text-white font-semibold">Intellegent Technology</span>. All rights reserved. 
              <span className="text-secondary"> By <a href="https://tarikul.dev/" target="_blank" rel="noopener noreferrer">TARIKUL ISLAM</a>.</span>
            </p>

            {/* Social Links */}
            <div className="flex items-center gap-3">
              {socialLinks.map((social) => (
                <a
                  key={social.label}
                  href={social.url}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="group glass-dark p-3 rounded-xl hover:bg-gradient-to-br hover:from-primary hover:to-secondary transition-all hover:scale-110"
                  aria-label={social.label}
                >
                  <social.icon size={18} className="group-hover:scale-110 transition-transform" />
                </a>
              ))}
            </div>
          </div>
        </div>
      </div>

      {/* Decorative Bottom Line */}
      <div className="h-1 bg-gradient-to-r from-primary via-secondary to-primary"></div>
    </footer>
  );
}
