"use client";

import { motion } from "framer-motion";
import { shouldAnimate } from "@/lib/motionUtils";

// Fade in animation - improved with smoother easing
export const FadeIn = ({
  children,
  delay = 0,
  duration = 0.6,
  className = "",
  useInitial = false,
  disableOnMobile = true,
}) => {
  const animate = shouldAnimate(disableOnMobile);

  if (useInitial) {
    return (
      <motion.div
        initial={{ opacity: animate ? 0 : 1, y: animate ? 30 : 0 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{
          duration: animate ? duration : 0,
          delay: animate ? delay : 0,
          ease: [0.25, 0.46, 0.45, 0.94], // Custom bezier for smooth motion
        }}
        className={className}
        style={{ overflow: "hidden" }}
      >
        {children}
      </motion.div>
    );
  }

  return (
    <motion.div
      initial={{ opacity: animate ? 0 : 1, y: animate ? 30 : 0 }}
      whileInView={{ opacity: 1, y: 0 }}
      viewport={{ once: true, amount: 0.3 }}
      transition={{
        duration: animate ? duration : 0,
        delay: animate ? delay : 0,
        ease: [0.25, 0.46, 0.45, 0.94],
      }}
      className={className}
      style={{ overflow: "hidden" }}
    >
      {children}
    </motion.div>
  );
};

// Slide in from left - reduced movement for mobile, smoother easing
export const SlideInLeft = ({
  children,
  delay = 0,
  duration = 0.7,
  className = "",
  useInitial = false,
  disableOnMobile = true,
}) => {
  const animate = shouldAnimate(disableOnMobile);

  if (useInitial) {
    return (
      <motion.div
        initial={{ opacity: animate ? 0 : 1, x: animate ? -30 : 0 }}
        animate={{ opacity: 1, x: 0 }}
        transition={{
          duration: animate ? duration : 0,
          delay: animate ? delay : 0,
          ease: [0.22, 1, 0.36, 1], // Smoother custom bezier
        }}
        className={className}
      >
        {children}
      </motion.div>
    );
  }

  return (
    <motion.div
      initial={{ opacity: animate ? 0 : 1, x: animate ? -30 : 0 }}
      whileInView={{ opacity: 1, x: 0 }}
      viewport={{ once: true, amount: 0.3 }}
      transition={{
        duration: animate ? duration : 0,
        delay: animate ? delay : 0,
        ease: [0.22, 1, 0.36, 1],
      }}
      className={className}
    >
      {children}
    </motion.div>
  );
};

// Slide in from right - reduced movement for mobile, smoother easing
export const SlideInRight = ({
  children,
  delay = 0,
  duration = 0.7,
  className = "",
  useInitial = false,
  disableOnMobile = true,
}) => {
  const animate = shouldAnimate(disableOnMobile);

  if (useInitial) {
    return (
      <motion.div
        initial={{ opacity: animate ? 0 : 1, x: animate ? 30 : 0 }}
        animate={{ opacity: 1, x: 0 }}
        transition={{
          duration: animate ? duration : 0,
          delay: animate ? delay : 0,
          ease: [0.22, 1, 0.36, 1],
        }}
        className={className}
      >
        {children}
      </motion.div>
    );
  }

  return (
    <motion.div
      initial={{ opacity: animate ? 0 : 1, x: animate ? 30 : 0 }}
      whileInView={{ opacity: 1, x: 0 }}
      viewport={{ once: true, amount: 0.3 }}
      transition={{
        duration: animate ? duration : 0,
        delay: animate ? delay : 0,
        ease: [0.22, 1, 0.36, 1],
      }}
      className={className}
    >
      {children}
    </motion.div>
  );
};

// Scale in animation - more subtle and professional
export const ScaleIn = ({
  children,
  delay = 0,
  duration = 0.6,
  className = "",
  useInitial = false,
  disableOnMobile = true,
}) => {
  const animate = shouldAnimate(disableOnMobile);

  if (useInitial) {
    return (
      <motion.div
        initial={{ opacity: animate ? 0 : 1, scale: animate ? 0.95 : 1 }}
        animate={{ opacity: 1, scale: 1 }}
        transition={{
          duration: animate ? duration : 0,
          delay: animate ? delay : 0,
          ease: [0.25, 0.46, 0.45, 0.94],
        }}
        className={className}
      >
        {children}
      </motion.div>
    );
  }

  return (
    <motion.div
      initial={{ opacity: animate ? 0 : 1, scale: animate ? 0.95 : 1 }}
      whileInView={{ opacity: 1, scale: 1 }}
      viewport={{ once: true, amount: 0.3 }}
      transition={{
        duration: animate ? duration : 0,
        delay: animate ? delay : 0,
        ease: [0.25, 0.46, 0.45, 0.94],
      }}
      className={className}
    >
      {children}
    </motion.div>
  );
};

// Stagger children animation - improved timing
export const StaggerContainer = ({
  children,
  staggerDelay = 0.12,
  className = "",
  useInitial = false,
  disableOnMobile = true,
}) => {
  const animate = shouldAnimate(disableOnMobile);

  if (useInitial) {
    return (
      <motion.div
        initial="hidden"
        animate="visible"
        variants={{
          visible: {
            transition: {
              staggerChildren: animate ? staggerDelay : 0,
              delayChildren: animate ? 0.1 : 0,
            },
          },
        }}
        className={className}
      >
        {children}
      </motion.div>
    );
  }

  return (
    <motion.div
      initial="hidden"
      whileInView="visible"
      viewport={{ once: true, amount: 0.2 }}
      variants={{
        visible: {
          transition: {
            staggerChildren: animate ? staggerDelay : 0,
            delayChildren: animate ? 0.1 : 0,
          },
        },
      }}
      className={className}
    >
      {children}
    </motion.div>
  );
};

export const StaggerItem = ({
  children,
  className = "",
  disableOnMobile = true,
}) => {
  const animate = shouldAnimate(disableOnMobile);

  return (
    <motion.div
      variants={{
        hidden: { opacity: animate ? 0 : 1, y: animate ? 25 : 0 },
        visible: {
          opacity: 1,
          y: 0,
          transition: {
            duration: animate ? 0.6 : 0,
            ease: [0.25, 0.46, 0.45, 0.94],
          },
        },
      }}
      className={className}
    >
      {children}
    </motion.div>
  );
};

// Hover scale animation - more professional and subtle
export const HoverScale = ({
  children,
  scale = 1.03,
  className = "",
  disableOnMobile = true,
}) => {
  const animate = shouldAnimate(disableOnMobile);

  return (
    <motion.div
      whileHover={{ scale: animate ? scale : 1 }}
      transition={{ duration: 0.25, ease: [0.25, 0.46, 0.45, 0.94] }}
      className={className}
    >
      {children}
    </motion.div>
  );
};

// Floating animation - smoother and more subtle
export const FloatingAnimation = ({
  children,
  className = "",
  disableOnMobile = true,
}) => {
  const animate = shouldAnimate(disableOnMobile);

  return (
    <motion.div
      animate={{
        y: animate ? [0, -8, 0] : 0,
      }}
      transition={{
        duration: 4,
        repeat: animate ? Infinity : 0,
        ease: [0.45, 0, 0.55, 1],
      }}
      className={className}
    >
      {children}
    </motion.div>
  );
};

// New: Blur fade in for premium feel
export const BlurFadeIn = ({
  children,
  delay = 0,
  duration = 0.7,
  className = "",
  useInitial = false,
  disableOnMobile = true,
}) => {
  const animate = shouldAnimate(disableOnMobile);

  if (useInitial) {
    return (
      <motion.div
        initial={{
          opacity: animate ? 0 : 1,
          filter: animate ? "blur(10px)" : "blur(0px)",
          y: animate ? 20 : 0,
        }}
        animate={{ opacity: 1, filter: "blur(0px)", y: 0 }}
        transition={{
          duration: animate ? duration : 0,
          delay: animate ? delay : 0,
          ease: [0.25, 0.46, 0.45, 0.94],
        }}
        className={className}
      >
        {children}
      </motion.div>
    );
  }

  return (
    <motion.div
      initial={{
        opacity: animate ? 0 : 1,
        filter: animate ? "blur(10px)" : "blur(0px)",
        y: animate ? 20 : 0,
      }}
      whileInView={{ opacity: 1, filter: "blur(0px)", y: 0 }}
      viewport={{ once: true, amount: 0.3 }}
      transition={{
        duration: animate ? duration : 0,
        delay: animate ? delay : 0,
        ease: [0.25, 0.46, 0.45, 0.94],
      }}
      className={className}
    >
      {children}
    </motion.div>
  );
};
