"use client";

import { useSearchParams } from "next/navigation";
import { useEffect, useState } from "react";
import {
  Search as SearchIcon,
  X,
  FileText,
  Box,
  Tag,
  Loader2,
} from "lucide-react";
import Link from "next/link";
import { useRouter } from "next/navigation";

export default function SearchResultsPage() {
  const searchParams = useSearchParams();
  const query = searchParams.get("q") || "";
  const router = useRouter();
  const [results, setResults] = useState([]);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    if (!query) {
      router.push("/");
      return;
    }

    const search = async () => {
      setIsLoading(true);
      try {
        const { searchAll } = await import("@/queries/searchQueries");
        const searchResults = await searchAll(query);
        setResults(searchResults || []);
      } catch (error) {
        console.error("Search error:", error);
        setResults([]);
      } finally {
        setIsLoading(false);
      }
    };

    search();
  }, [query, router]);

  const getResultIcon = (type) => {
    switch (type) {
      case "blog":
        return <FileText className="w-5 h-5 text-gray-500" />;
      case "product":
        return <Box className="w-5 h-5 text-blue-500" />;
      case "category":
        return <Tag className="w-5 h-5 text-green-500" />;
      default:
        return <SearchIcon className="w-5 h-5 text-gray-400" />;
    }
  };

  const getResultUrl = (result) => {
    if (!result) return "#";
    if (result.url) return result.url;
    if (result.type === "blog" && result.slug) return `/blog/${result.slug}`;
    if (result.type === "product" && result.slug)
      return `/product/${result.slug}`;
    if (result.type === "category" && result.slug)
      return `/products/${result.slug}`;
    return "#";
  };

  if (!query) return null;

  return (
    <div className="container mx-auto px-4 py-8 max-w-4xl">
      <div className="mb-8">
        <h1 className="text-2xl font-bold text-gray-900 mb-2">
          Search Results for "{query}"
        </h1>
        <p className="text-gray-500">
          {isLoading ? "Searching..." : `${results.length} results found`}
        </p>
      </div>

      {isLoading ? (
        <div className="flex justify-center items-center h-64">
          <div className="flex flex-col items-center space-y-4">
            <Loader2 className="animate-spin h-12 w-12 text-blue-500" />
            <p className="text-gray-600">Searching...</p>
          </div>
        </div>
      ) : results.length > 0 ? (
        <div className="space-y-6">
          {results.map((result) => (
            <Link
              key={`${result.type}-${result.id}`}
              href={getResultUrl(result)}
              className="block p-4 border border-gray-200 rounded-lg hover:shadow-md transition-shadow"
            >
              <div className="flex items-start">
                <div className="flex-shrink-0 mr-4">
                  {result.type === "product" && result.image ? (
                    <div className="w-16 h-16 relative rounded-md overflow-hidden bg-gray-100">
                      <img
                        src={result.image}
                        alt={result.name || "Product"}
                        className="w-full h-full object-cover"
                      />
                    </div>
                  ) : (
                    <div className="w-16 h-16 rounded-md bg-gray-50 flex items-center justify-center">
                      {getResultIcon(result.type)}
                    </div>
                  )}
                </div>
                <div className="flex-1">
                  <h3 className="text-lg font-medium text-gray-900 mb-1">
                    {result.title || result.name || "Untitled"}
                  </h3>
                  {result.description && (
                    <p className="text-gray-600 mb-2 line-clamp-2">
                      {result.description}
                    </p>
                  )}
                  <div className="flex items-center">
                    <span
                      className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                        result.type === "blog"
                          ? "bg-purple-100 text-purple-800"
                          : result.type === "product"
                          ? "bg-blue-100 text-blue-800"
                          : "bg-green-100 text-green-800"
                      }`}
                    >
                      {result.type.charAt(0).toUpperCase() +
                        result.type.slice(1)}
                    </span>
                    {/* {result.price && (
                      <span className="ml-2 text-sm font-medium text-gray-900">
                        ${!result.isPriceHide && result.price}
                      </span>
                    )} */}
                  </div>
                </div>
              </div>
            </Link>
          ))}
        </div>
      ) : (
        <div className="text-center py-12">
          <SearchIcon className="w-12 h-12 mx-auto text-gray-300 mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-1">
            No results found
          </h3>
          <p className="text-gray-500">
            We couldn't find any results for "{query}". Try different keywords.
          </p>
        </div>
      )}
    </div>
  );
}
