'use client';

import React, { useState, useMemo } from 'react';
import Image from 'next/image';
import { ChevronLeft, ChevronRight, ZoomIn } from 'lucide-react';

export default function ProductImageGallery({ images: initialImages, productName, mainImage }) {
  // Combine mainImage with images array if mainImage exists and is not already in the images array
  const images = useMemo(() => {
    if (!mainImage) return initialImages || [];
    if (initialImages && initialImages.length > 0) {
      // If mainImage is already in the images array, return as is
      if (initialImages.includes(mainImage)) return initialImages;
      // Otherwise, add mainImage as the first image
      return [mainImage, ...initialImages];
    }
    return [mainImage];
  }, [initialImages, mainImage]);

  const [currentImageIndex, setCurrentImageIndex] = useState(0);
  const [isZoomed, setIsZoomed] = useState(false);

  const nextImage = () => {
    setCurrentImageIndex((prevIndex) => 
      prevIndex === images.length - 1 ? 0 : prevIndex + 1
    );
  };

  const prevImage = () => {
    setCurrentImageIndex((prevIndex) =>
      prevIndex === 0 ? images.length - 1 : prevIndex - 1
    );
  };

  const selectImage = (index) => {
    setCurrentImageIndex(index);
  };

  const toggleZoom = () => {
    setIsZoomed(!isZoomed);
  };

  // Use the first image as fallback if images array is empty
  const displayImages = images && images.length > 0 ? images : [];

  if (displayImages.length === 0) {
    return (
      <div className="w-full bg-gray-100 rounded-lg aspect-square flex items-center justify-center">
        <span className="text-gray-400">No images available</span>
      </div>
    );
  }

  return (
    <div className="w-full">
      {/* Main Image */}
      <div className="relative bg-white rounded-lg overflow-hidden shadow-lg mb-4 aspect-square">
        <div 
          className={`relative w-full h-full transition-transform duration-300 ease-in-out ${isZoomed ? 'scale-125' : 'scale-100'}`}
          onClick={toggleZoom}
        >
          <Image
            src={displayImages[currentImageIndex]}
            alt={`${productName} - Image ${currentImageIndex + 1}`}
            fill
            className="object-contain cursor-zoom-in"
            sizes="(max-width: 768px) 100vw, 50vw"
            priority
          />
          <div className="absolute top-2 right-2 bg-black/50 text-white p-1.5 rounded-full">
            <ZoomIn size={20} />
          </div>
        </div>
        
        {/* Navigation Arrows - Only show if there are multiple images */}
        {displayImages.length > 1 && (
          <>
            <button
              onClick={(e) => {
                e.stopPropagation();
                prevImage();
              }}
              className="absolute left-2 top-1/2 -translate-y-1/2 bg-white/80 hover:bg-white text-gray-800 p-2 rounded-full shadow-md transition-all"
              aria-label="Previous image"
            >
              <ChevronLeft size={24} />
            </button>
            <button
              onClick={(e) => {
                e.stopPropagation();
                nextImage();
              }}
              className="absolute right-2 top-1/2 -translate-y-1/2 bg-white/80 hover:bg-white text-gray-800 p-2 rounded-full shadow-md transition-all"
              aria-label="Next image"
            >
              <ChevronRight size={24} />
            </button>
          </>
        )}
      </div>

      {/* Thumbnails - Only show if there are multiple images */}
      {displayImages.length > 1 && (
        <div className="flex space-x-2 overflow-x-auto py-2">
          {displayImages.map((image, index) => (
            <button
              key={index}
              onClick={() => selectImage(index)}
              className={`flex-shrink-0 w-16 h-16 rounded-md overflow-hidden border-2 ${
                currentImageIndex === index 
                  ? 'border-blue-500 ring-2 ring-blue-200' 
                  : 'border-transparent hover:border-gray-300'
              } transition-all`}
              aria-label={`View image ${index + 1} of ${displayImages.length}`}
            >
              <Image
                src={image}
                alt={`Thumbnail ${index + 1}`}
                width={64}
                height={64}
                className="w-full h-full object-cover"
              />
            </button>
          ))}
        </div>
      )}
    </div>
  );
}
